/*
	Raytracer stuff for Turbo C++ 3.0
	Written/Converted by Matt Pierce
*/

#include <stdio.h>

#ifndef _3DTRACE_H
#define _3DTRACE_H

#define UID_SPHERE 0
#define UID_PLANE 1

struct STexture {
	unsigned char *data;
	int width, height;
};

struct SContactInfo {
	float t;
	unsigned int uid;
	unsigned int objtype;
	SVect3d globalp;
	SVect3d localp;
	SVect3d globaln;
	SVect3d surface;
	unsigned char col;
	unsigned char lightammount;
	unsigned char reflectcol;
	unsigned char specularcol;
	STexture *tex;
	unsigned char tx, ty;
};

struct S3dObject {
	unsigned int objtype;
	unsigned int uid;
	float far *mat;
	float far *imat;
	SVect3d surface;
	unsigned char col;
	STexture *tex;
	SVect3d tp, tz, gz;
	float util_float_0, util_float_1;
};

struct S3dLight {
	SVect3d pos;
	unsigned char col;
};

void TracePlane(S3dObject *obj, const SVect3d &op, const SVect3d &od,
		SContactInfo *contact)
{
	SVect3d p, d, zero;
	zero.x = zero.y = zero.z = 0.0f;
//	p = Transform(op, obj->imat);
//	d = Transform(od, obj->imat);
	MacroTransform(op, obj->imat, p);
	MacroTransform(od, obj->imat, d);
	zero = Transform(zero, obj->imat);
	d.x -= zero.x;
	d.y -= zero.y;
	d.z -= zero.z;

	if (d.y == 0.0f) {
		contact->t = -1.0f;
	} else {
		contact->t = -p.y/d.y;
		if (contact->t > 0.0f ) {
			contact->uid = obj->uid;
			contact->objtype = obj->objtype;
			contact->globalp = op + od*contact->t;
			contact->localp = p + d*contact->t;
			zero.x = 0.0f;
			zero.y = 0.0f;
			zero.z = 0.0f;
			zero = Transform(zero, obj->mat);
			contact->globaln.x = 0.0f;
			contact->globaln.y = 1.0f;
			contact->globaln.z = 0.0f;
			contact->globaln = Transform(contact->globaln, obj->mat);
			contact->globaln.x -= zero.x;
			contact->globaln.y -= zero.y;
			contact->globaln.z -= zero.z;
			contact->surface = obj->surface;
			contact->col = obj->col;
			contact->tex = obj->tex;
		}
	}
}

void TracePlaneFirstHit(S3dObject *obj, const SVect3d &op, const SVect3d &od,
		SContactInfo *contact)
{
	SVect3d d, localn;
	// d = Transform(od, obj->imat);
	MacroTransform(od, obj->imat, d);
	d.x -= obj->tz.x;
	d.y -= obj->tz.y;
	d.z -= obj->tz.z;

	if (d.y == 0.0f) {
		contact->t = -1.0f;
	} else {
		contact->t = -obj->tp.y/d.y;
		if (contact->t > 0.0f ) {
			contact->uid = obj->uid;
			contact->objtype = obj->objtype;
			contact->globalp = op + od*contact->t;
			contact->localp = obj->tp + d*contact->t;
			localn.x = 0.0f;
			localn.y = 1.0f;
			localn.z = 0.0f;
			MacroTransform(localn, obj->mat, contact->globaln);
//			contact->globaln = Transform(contact->globaln, obj->mat);
			contact->globaln.x -= obj->gz.x;
			contact->globaln.y -= obj->gz.y;
			contact->globaln.z -= obj->gz.z;
			contact->surface = obj->surface;
			contact->col = obj->col;
			contact->tex = obj->tex;
		}
	}
}

float TracePlaneShadow(S3dObject *obj, const SVect3d &op, const SVect3d &od)
{
	SVect3d p, d;

	p = Transform(op, obj->imat);
	d = Transform(od+op, obj->imat) - p;

	if (d.y == 0.0f) return -1.0f;
	else return -p.y/d.y;
}

void TraceSphere(S3dObject *obj, const SVect3d &op, const SVect3d &od,
		SContactInfo *contact)
{
	float t1, t2;
	SVect3d p, d, zero;
	zero.x = zero.y = zero.z = 0.0f;
	p = Transform(op, obj->imat);
	d = Transform(od, obj->imat);
	zero = Transform(zero, obj->imat);
	d.x -= zero.x;
	d.y -= zero.y;
	d.z -= zero.z;

	float a = MacroDotProd(d, d);
	float b = 2.0f*(MacroDotProd(p, d));
	float c = MacroDotProd(p, p) - 1.0f;
	float det = b*b - 4.0f*a*c;

	if (det < 0.0f) {
		contact->t = -1.0f;
	} else {
		det = sqrt(det);
		a *= 2.0f;
		t1 = (-b-det)/a;
		t2 = (-b+det)/a;
		if (t1>0.0f) contact->t = t1;
		else contact->t = t2;
		if (contact->t >= 0.0f) {
			contact->uid = obj->uid;
			contact->objtype = obj->objtype;
			contact->globalp = op + od*contact->t;
			contact->localp = p + d*contact->t;
			zero.x = 0.0f;
			zero.y = 0.0f;
			zero.z = 0.0f;
			zero = Transform(zero, obj->mat);
			contact->globaln = Transform(contact->localp, obj->mat);
			contact->globaln.x -= zero.x;
			contact->globaln.y -= zero.y;
			contact->globaln.z -= zero.z;
			contact->surface = obj->surface;
			contact->col = obj->col;
			contact->tex = obj->tex;
		}
	}
}

void TraceSphereFirstHit(S3dObject *obj, const SVect3d &op, const SVect3d &od,
		SContactInfo *contact)
{
	float t1, t2;
	SVect3d d;
//	d = Transform(od, obj->imat);
	MacroTransform(od, obj->imat, d);
	d.x -= obj->tz.x;
	d.y -= obj->tz.y;
	d.z -= obj->tz.z;

	float a = MacroDotProd(d, d);
	float b = 2.0f*(MacroDotProd(obj->tp, d));
	float det = b*b - 4.0f*a*obj->util_float_0;

	if (det < 0.0f) {
		contact->t = -1.0f;
	} else {
		det = sqrt(det);
		a *= 2.0f;
		t1 = (-b-det)/a;
		t2 = (-b+det)/a;
		if (t1>0.0f) contact->t = t1;
		else contact->t = t2;
		if (contact->t >= 0.0f) {
			contact->uid = obj->uid;
			contact->objtype = obj->objtype;
			contact->globalp = op + od*contact->t;
			contact->localp = obj->tp + d*contact->t;
			MacroTransform(contact->localp, obj->mat, contact->globaln);
			contact->globaln.x -= obj->gz.x;
			contact->globaln.y -= obj->gz.y;
			contact->globaln.z -= obj->gz.z;
			contact->surface = obj->surface;
			contact->col = obj->col;
			contact->tex = obj->tex;
		}
	}
}

float TraceSphereShadow(S3dObject *obj, const SVect3d &op, const SVect3d &od)
{
	SVect3d p, d, s;
	s = op+od;
	//p = Transform(op, obj->imat);
	//d = Transform(op+od, obj->imat) - p;
	MacroTransform(op, obj->imat, p);
	MacroTransform(s, obj->imat, d);
	d.x -= p.x;
	d.y -= p.y;
	d.z -= p.z;

	float a = 2.0f*(MacroDotProd(d, d));
	float b = 2.0f*(MacroDotProd(p, d));
	float c = MacroDotProd(p, p) - 1.0f;
	float det = b*b - 2.0f*a*c;

	if (det<0.0f) return -1.0f;
	det = sqrt(det);
	float t = (-b-det)/a;
	if (t>=0.0f) return t;
	t = (-b+det)/a;
	return t;

}

void TraceObject(S3dObject *obj, const SVect3d &op, const SVect3d &od,
		SContactInfo *contact)
{
	switch (obj->objtype) {
		case UID_SPHERE:
			TraceSphere(obj, op, od, contact);
			break;
		case UID_PLANE:
			TracePlane(obj, op, od, contact);
			break;
		default:
			contact->t = -1.0f;
			break;
	}
}

void TraceObjectFirstHit(S3dObject *obj, const SVect3d &op, const SVect3d &od,
		SContactInfo *contact)
{
	switch (obj->objtype) {
		case UID_SPHERE:
			TraceSphereFirstHit(obj, op, od, contact);
			break;
		case UID_PLANE:
			TracePlaneFirstHit(obj, op, od, contact);
			break;
		default:
			contact->t = -1.0f;
			break;
	}
}

float TraceObjectShadow(S3dObject *obj, const SVect3d &op, const SVect3d &od)
{
	switch (obj->objtype) {
		case UID_SPHERE:
			return TraceSphereShadow(obj, op, od);
		case UID_PLANE:
			return TracePlaneShadow(obj, op, od);
		default:
			return -1.0f;
	}
}

void InitObject(S3dObject *obj)
{
	obj->mat = new float[16];
	obj->imat = new float[16];
	LoadIdentity(obj->mat);
	LoadIdentity(obj->imat);
	obj->surface.x = 1.0f;
	obj->surface.y = 0.0f;
	obj->surface.z = 0.0f;
	obj->col = 255;
	obj->uid = 0;
	obj->objtype = 0;
	obj->tex = NULL;
}

void PrepFirstHit(S3dObject *obj, const SVect3d &p)
{
	obj->tp = Transform(p, obj->imat);
	obj->tz.x = obj->tz.y = obj->tz.z = 0;
	obj->tz = Transform(obj->tz, obj->imat);
	obj->gz.x = obj->gz.y = obj->gz.z = 0;
	obj->gz = Transform(obj->gz, obj->mat);
	switch(obj->objtype) {
		case UID_SPHERE:
			obj->util_float_0 = MacroDotProd(obj->tp, obj->tp) - 1.0f;
			break;
		default:
			break;
	}
}

inline unsigned char GetTexColPlane(const SVect3d &p, STexture *tex)
{
	int tx, ty;
	int w, h;
	w = tex->width;
	h = tex->height;

	if (p.x<0.0f) tx = w - w*((int)p.x - p.x);
	else tx = (int)(p.x*w)%w;

	if (p.z<0.0f) ty = h - h*((int)p.z - p.z);
	else ty = (int)(p.z*h)%h;

	return tex->data[ty*w+tx];
}

inline unsigned char GetTexColSphere(const SVect3d &p, STexture *tex)
{
	return GetTexColPlane(p, tex);
}

unsigned char GetTexCol(const unsigned int &objtype, const SVect3d &p, STexture *tex)
{
	switch(objtype) {
		case UID_PLANE:
			return GetTexColPlane(p, tex);
		case UID_SPHERE:
			return GetTexColSphere(p, tex);
		default:
			return 255;
	}
}

void LoadTexture(char *filename, int a_w, int a_h, STexture *tex)
{
	tex->width = a_w;
	tex->height = a_h;
	tex->data = new unsigned char[a_w*a_h];

	FILE *my_file;
	my_file = fopen(filename, "rb");
	fread((void *)(tex->data), 1, a_w*a_h, my_file);
	fclose(my_file);

}



#endif