// Generic Model Mesh object class
// Matt Pierce
// March 2002
// This was created to help speed along a computer graphics lab.
// It is in prototype stagees, but works enough for this (and
// probably the next few) lab.

// New Features:  (first edition, all features are new
//		Cel Shading (set by a single flag)
//		Texture Mapping
//		tons of useful vector and matrix functions
//		Loading Mesh from file with just a specified name
//		Casts a shadow of the mesh on a plane in real world coords
//		Flat shading safe mode

#ifndef GENERIC_POLY_H
#define GENERIC_POLY_H


#include <iostream.h>
#include <fstream.h>
#include <math.h>
#include <gl/glut.h>

// Defines all flags.  These are bitwise, so they can
// be stored in one unsigned int
const int GENERIC_LOAD_NORMALS = 1 <<0;
const int GENERIC_LOAD_TEX_COORDS = 1 << 1;
const int GENERIC_DRAW_TEX = 1 << 2;
const int GENERIC_DRAW_NORM = 1 << 3;
const int GENERIC_CEL_SHADE = 1 << 4;
const int GENERIC_DRAW_FLAT = 1 << 5;

GLuint generic_cel_tex_index[1];		// holds cel-shading index
										// for texture binding
int generic_cel_flag = 0;				// tells if any generic poly
										// has initialized the cel-shading
										// texture

// The raw data for the Cel Shading texture
float cel_tex[32][3] = {
	0.2, 0.2, 0.2,
	0.2, 0.2, 0.2,
	0.2, 0.2, 0.2,
	0.2, 0.2, 0.2,
	0.5, 0.5, 0.5,
	0.5, 0.5, 0.5,
	0.5, 0.5, 0.5,
	0.5, 0.5, 0.5,
	0.5, 0.5, 0.5,
	0.5, 0.5, 0.5,
	0.5, 0.5, 0.5,
	0.5, 0.5, 0.5,
	0.5, 0.5, 0.5,
	0.5, 0.5, 0.5,
	1.0, 1.0, 1.0,
	1.0, 1.0, 1.0,
	1.0, 1.0, 1.0,
	1.0, 1.0, 1.0,
	1.0, 1.0, 1.0,
	1.0, 1.0, 1.0,
	1.0, 1.0, 1.0,
	1.0, 1.0, 1.0,
	1.0, 1.0, 1.0,
	1.0, 1.0, 1.0,
	1.0, 1.0, 1.0,
	1.0, 1.0, 1.0,
	1.0, 1.0, 1.0,
	1.0, 1.0, 1.0,
	1.0, 1.0, 1.0,
	1.0, 1.0, 1.0,
	1.0, 1.0, 1.0,
	1.0, 1.0, 1.0};

/// Handy Matrix and Vector tools from NeHe Tutorial #38
// http://nehe.gamedev.net
typedef struct tagMATRIX										// A Structure To Hold An OpenGL Matrix ( NEW )
{
	float Data[16];												// We Use [16] Due To OpenGL's Matrix Format ( NEW )
}
MATRIX;
	
struct vector {
	float x,y,z;
};

vector RotateVector (MATRIX &M, vector V)				// Rotate A Vector Using The Supplied Matrix ( NEW )
{
	vector D;
	D.x = (M.Data[0] * V.x) + (M.Data[4] * V.y) + (M.Data[8]  * V.z);	// Rotate Around The X Axis ( NEW )
	D.y = (M.Data[1] * V.x) + (M.Data[5] * V.y) + (M.Data[9]  * V.z);	// Rotate Around The Y Axis ( NEW )
	D.z = (M.Data[2] * V.x) + (M.Data[6] * V.y) + (M.Data[10] * V.z);	// Rotate Around The Z Axis ( NEW )
	return D;
}

// this one's mine, based on NeHe
// Doesn't seem to do anything different, though.
vector MatrixMult (MATRIX &M, vector V)				// Rotate A Vector Using The Supplied Matrix ( NEW )
{
	vector D;
	float id;
	D.x = (M.Data[0] * V.x) + (M.Data[4] * V.y) + (M.Data[8]  * V.z) + M.Data[12];	// Rotate Around The X Axis ( NEW )
	D.y = (M.Data[1] * V.x) + (M.Data[5] * V.y) + (M.Data[9]  * V.z) + M.Data[13];	// Rotate Around The Y Axis ( NEW )
	D.z = (M.Data[2] * V.x) + (M.Data[6] * V.y) + (M.Data[10] * V.z) + M.Data[14];	// Rotate Around The Z Axis ( NEW )
	id =  (M.Data[3] * V.x) + (M.Data[7] * V.y) + (M.Data[11] * V.z) + M.Data[15];
	D.x /= id;
	D.y /= id;
	D.z /= id;
	return D;
}

vector vectNormalize(vector inVect)
{
	float dist = sqrt(inVect.x*inVect.x + inVect.y*inVect.y + inVect.z*inVect.z);
	vector returnVect;
	returnVect.x = inVect.x / dist;
	returnVect.y = inVect.y / dist;
	returnVect.z = inVect.z / dist;
	return returnVect;
}

float gmmDotProduct (vector V1, vector V2)				// Calculate The Angle Between The 2 Vectors ( NEW )
{
	return V1.x * V2.x + V1.y * V2.y + V1.z * V2.z;				// Return The Angle ( NEW )
}

vector gmmCrossProduct (vector a, vector b)
{
	vector c;
	c.x = a.y*b.z - a.z*b.y;
	c.y = a.z*b.x - a.x*b.z;
	c.z = a.x*b.y - a.y*b.x;
	return c;
}

// End of NeHe vector math adaptations


// This structure is a genericPoly
// it can hold just about any type of poly, completely independent of other polys.
struct genericPoly {
	vector p1, p2, p3;
	vector n1, n2, n3;
	vector t1, t2, t3;
	unsigned int flags; };


// genericModelMesh object declaration
class genericModelMesh {
private:
	genericPoly *genericPolyList;		// Holds genericPoly data for object mesh
	unsigned int textureID;				// Holds textureID for binding
										// (supplied by user)
	unsigned int numberPolys;			// Keeps track of number of genericPoly
	unsigned int polyAttributes;		// Bitwise flags of polygon attributes
	void genCelTex(void);				// Function to generate cel-shading texture
	vector lightAngle;					// the angle used for cel-shading & shadows

public:
	genericModelMesh(void);				// Constructor
	genericModelMesh(char* filename);	// Constructor
	~genericModelMesh();				// Destructor
	void DrawModel();					// Draws object to screen
	genericPoly* GetListCopy();			// Gives user a fresh copy of genericPolyList
	void LoadModel(char* filename);		// Loads a mesh from a file
	void AssignTexture (unsigned int newTexture);	// Assigns a texture (user must do loading now)
	void SetFlag(unsigned int newFlag);		// Sets bitwise flags
	void UnsetFlag(unsigned int newFlag);	// Clears bitwise flags
	void outputSome(void);					// uses cout to describe what's going on (debug func)
	void ProjectShadow(float, float, float, float);	// casts shadow onto a plane.
};

// void genCelTex(void)
// takes cel shading texture and puts it into OpenGL subsystem
void genericModelMesh::genCelTex(void)
{
	// Enable Texturing
	glEnable(GL_TEXTURE_1D);
	// Get a texture index
	glGenTextures(1, &generic_cel_tex_index[0]);
	// bind that index
	glBindTexture(GL_TEXTURE_1D, generic_cel_tex_index[0]);
	// Set preferences
	glTexParameteri (GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);	
	glTexParameteri (GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	// Upload image from array to OpenGL subsystem
	glTexImage1D (GL_TEXTURE_1D, 0, GL_RGB, 32, 0, GL_RGB , GL_FLOAT, cel_tex);
	// Mark this task as done
	generic_cel_flag = 1;
}

// genericModelMesh(void)
// initializes an object mesh
genericModelMesh::genericModelMesh(void)
{
	textureID = 0;
	numberPolys = 0;
	lightAngle.x = 1;
	lightAngle.y = 1;
	lightAngle.z = 2;
	lightAngle = vectNormalize(lightAngle);
}

// genericModelMesh(char* filename)
// initializes an object mesh and loads a file
genericModelMesh::genericModelMesh(char* filename)
{
	textureID = 0;
	numberPolys = 0;
	lightAngle.x = 1;
	lightAngle.y = 1;
	lightAngle.z = 2;
	lightAngle = vectNormalize(lightAngle);
	LoadModel(filename);
}

// ~genericModelMesh()
// cleans up all memory used by the mesh object
genericModelMesh::~genericModelMesh()
{
	if (genericPolyList != NULL)
		delete [] genericPolyList;
}

// void outputSome(void)
// puts some data to the console,
// handy for debugging
void genericModelMesh::outputSome(void)
{
	unsigned int x;
	for (x=0;x<10;x++)
	{
		cout << genericPolyList[x].p1.x << ", " << genericPolyList[x].p1.y << ", " << genericPolyList[x].p1.z << '\n';
		cout << genericPolyList[x].p2.x << ", " << genericPolyList[x].p2.y << ", " << genericPolyList[x].p2.z << '\n';
		cout << genericPolyList[x].p3.x << ", " << genericPolyList[x].p3.y << ", " << genericPolyList[x].p3.z << '\n';
		cout << genericPolyList[x].n1.x << ", " << genericPolyList[x].n1.y << ", " << genericPolyList[x].n1.z << '\n';
		cout << genericPolyList[x].n2.x << ", " << genericPolyList[x].n2.y << ", " << genericPolyList[x].n2.z << '\n';
		cout << genericPolyList[x].n3.x << ", " << genericPolyList[x].n3.y << ", " << genericPolyList[x].n3.z << '\n';
	}
	cout << polyAttributes << ", " << numberPolys << '\n';
	if(polyAttributes & GENERIC_DRAW_TEX) cout << "Texture Enabled\n";
	if(polyAttributes & GENERIC_DRAW_NORM) cout << "Normals Enabled\n";
	if(polyAttributes & GENERIC_CEL_SHADE) cout << "Cel Shading Enabled\n";
	cout << cel_tex[4][1] << ' ' << cel_tex[4][2] << ' ' << cel_tex[4][3];
}

// genericPoly* GetListCopy(void)
// allocates space for a copy of the list,
// copies it, and returns that pointer
// to the user. Allows info to be read
// directly, without the user interfering with it.
genericPoly *genericModelMesh::GetListCopy(void)
{
	genericPoly *temp = new genericPoly[numberPolys];
	unsigned int x;
	for (x=0;x<numberPolys; x++)
	{
		temp[x].p1 = genericPolyList[x].p1;
		temp[x].p2 = genericPolyList[x].p2;
		temp[x].p3 = genericPolyList[x].p3;
		temp[x].n1 = genericPolyList[x].n1;
		temp[x].n2 = genericPolyList[x].n2;
		temp[x].n3 = genericPolyList[x].n3;
		temp[x].t1 = genericPolyList[x].t1;
		temp[x].t2 = genericPolyList[x].t2;
		temp[x].t3 = genericPolyList[x].t3;
	}
	return temp;
}

// void SetFlag(unsigned int newFlag)
// sets the bit for a flag... usually used with mnemonics
// ex.: SetFlag(GENERIC_CEL_SHADE);
void genericModelMesh::SetFlag(unsigned int newFlag)
{
	polyAttributes |= newFlag;
}

// void UnsetFlag(unsigned int newFlag)
// clears the bit for a flag .. usually used with mnemonics
// ex.: UnsetFlag(GENERIC_DRAW_NORM);
void genericModelMesh::UnsetFlag(unsigned int newFlag)
{
	if (polyAttributes & newFlag) polyAttributes ^= newFlag;
}

// void AssignTexture(unsigned int newTexture)
// gets a texture designator index from the user
// and sets that for the current object.
//		Note: nothing will be displayed until the user
//		also calls object.SetFlag(GENERIC_DRAW_TEX);
void genericModelMesh::AssignTexture(unsigned int newTexture)
{
	textureID = newTexture;
}

// void DrawModel(void)
// draws the object mesh to the screen
void genericModelMesh::DrawModel(void)
{
	// If nobody has initialized the cel shading texture, initialize it.
	if (generic_cel_flag == 0) genCelTex();
	unsigned int x;				// Loop counter
	MATRIX rotationMatrix;		// Matrix holder (for cel shading)
	vector tmpNormal;			// a temporary normal vector (for cel shading)
	float tmpShade;				// the 1D texture coordinate (for cel shading)

	// Get a copy of OpenGL's matrix
	glGetFloatv (GL_MODELVIEW_MATRIX, rotationMatrix.Data);

	// if we're cel shading, disable built in lighting
	if (polyAttributes & GENERIC_CEL_SHADE)
		glDisable(GL_LIGHTING);
	// if we're texturing, enable it.
	if (polyAttributes & GENERIC_DRAW_TEX)
		glBindTexture(GL_TEXTURE_2D, textureID);
	
	// start the geometry
	glPushMatrix();
	for (x=0;x<numberPolys;x++)
	{
		// our stuff will be triangles all the way, so we start one
		tmpNormal.x = genericPolyList[x].n1.x + genericPolyList[x].n2.x + genericPolyList[x].n3.x;
		tmpNormal.y = genericPolyList[x].n1.y + genericPolyList[x].n2.y + genericPolyList[x].n3.y;
		tmpNormal.z = genericPolyList[x].n1.z + genericPolyList[x].n2.z + genericPolyList[x].n3.z;
		tmpNormal = vectNormalize(tmpNormal);

		glBegin(GL_TRIANGLES);
			// if we're texturing, set coordinates
			if (polyAttributes & GENERIC_DRAW_TEX)
				glTexCoord2f(genericPolyList[x].t1.x, genericPolyList[x].t1.y);
			// if we've got normals, assign them
			if (polyAttributes & GENERIC_DRAW_NORM)
			{
				if (polyAttributes & GENERIC_DRAW_FLAT)
					glNormal3f(tmpNormal.x, tmpNormal.y, tmpNormal.z);
				else
					glNormal3f(genericPolyList[x].n1.x, genericPolyList[x].n1.y, genericPolyList[x].n1.z);
			}
			// put the point
			glVertex3f(genericPolyList[x].p1.x, genericPolyList[x].p1.y, genericPolyList[x].p1.z);
	
			// if we're texturing, set coordinates
			if (polyAttributes & GENERIC_DRAW_TEX)
				glTexCoord2f(genericPolyList[x].t2.x, genericPolyList[x].t2.y);
			// if we've got normals, assign them
			if (polyAttributes & GENERIC_DRAW_NORM)
			{
				if (polyAttributes & GENERIC_DRAW_FLAT)
					glNormal3f(tmpNormal.x, tmpNormal.y, tmpNormal.z);
				else
					glNormal3f(genericPolyList[x].n2.x, genericPolyList[x].n2.y, genericPolyList[x].n2.z);
			}
			// put the point
			glVertex3f(genericPolyList[x].p2.x, genericPolyList[x].p2.y, genericPolyList[x].p2.z);
			
			// if we're texturing, set coordinates
			if (polyAttributes & GENERIC_DRAW_TEX)
				glTexCoord2f(genericPolyList[x].t3.x, genericPolyList[x].t3.y);
			// if we've got normals, assign them
			if (polyAttributes & GENERIC_DRAW_NORM)
			{
				if (polyAttributes & GENERIC_DRAW_FLAT)
					glNormal3f(tmpNormal.x, tmpNormal.y, tmpNormal.z);
				else
					glNormal3f(genericPolyList[x].n3.x, genericPolyList[x].n3.y, genericPolyList[x].n3.z);
			}
			// put the point
			glVertex3f(genericPolyList[x].p3.x, genericPolyList[x].p3.y, genericPolyList[x].p3.z);
		glEnd();
			// If we're doing cel-shading, we have to start another triangle to lay on top
			// of the last one we rendered
			if (polyAttributes & GENERIC_CEL_SHADE)
			{
				// Set up the flags for cel shading
				glEnable(GL_BLEND);
				glEnable(GL_TEXTURE_1D);
				glBindTexture(GL_TEXTURE_1D, generic_cel_tex_index[0]);
				// this is so we shade what we just drew, not totally blank it.
				glDepthFunc(GL_LEQUAL);
				glBlendFunc(GL_DST_COLOR, GL_ZERO);

				glBegin(GL_TRIANGLES);
			
					// set the normal to our original, but roatated according to the current matrix
					tmpNormal = RotateVector(rotationMatrix, genericPolyList[x].n1);
					// normalize the normal
					tmpNormal = vectNormalize(tmpNormal);
					// find the difference between the light angle and our adjusted normal
					tmpShade = gmmDotProduct(lightAngle, tmpNormal);
					if (tmpShade < 0.01) tmpShade = 0.01;
					// Since texture coordinates are represented as 0.0 - 1.0, we can just
					// assign the coordinate directly
					glTexCoord1f(tmpShade);
					// put the point
					glVertex3f(genericPolyList[x].p1.x, genericPolyList[x].p1.y, genericPolyList[x].p1.z);
			
					// do it again for second point of triangle
					tmpNormal = RotateVector(rotationMatrix, genericPolyList[x].n2);
					tmpNormal = vectNormalize(tmpNormal);
					tmpShade = gmmDotProduct(lightAngle, tmpNormal);
					if (tmpShade < 0.01) tmpShade = 0.01;
					glTexCoord1f(tmpShade);
					glVertex3f(genericPolyList[x].p2.x, genericPolyList[x].p2.y, genericPolyList[x].p2.z);
			
					// and one more time for the third
					tmpNormal = RotateVector(rotationMatrix, genericPolyList[x].n3);
					tmpNormal = vectNormalize(tmpNormal);
					tmpShade = gmmDotProduct(lightAngle, tmpNormal);
					if (tmpShade < 0.01) tmpShade = 0.01;
					glTexCoord1f(tmpShade);
					glVertex3f(genericPolyList[x].p3.x, genericPolyList[x].p3.y, genericPolyList[x].p3.z);
				glEnd();
				// set things back to normal
				glDisable(GL_TEXTURE_1D);
				glDisable(GL_BLEND);
			}
	}
	glEnd();
	glPopMatrix();
	// if we changed this by going into cel shading, set it back
	// if (polyAttributes & GENERIC_CEL_SHADE) glEnable(GL_LIGHTING);
	glEnable(GL_LIGHTING);
}

// void ProjectShadow(float, float, float, float)
// Draws the object entirely in black
// and projected onto an infinite plane
// A,B,C: the normal of the plane
// D: the distance from the origin
// The geometric plane is defined as:
//		Ax + By + Cz + D = 0
// A ray of light cast through a point yeilds
//		p(t) = P + L*t
//			where P is the point, L is the normalized light vector
//			t is the distance along the line, and p is the output vector
void genericModelMesh::ProjectShadow(float A, float B, float C, float D)
{
	// We have to do our own matrix rotation here
	MATRIX rotationMatrix;
	glGetFloatv (GL_MODELVIEW_MATRIX, rotationMatrix.Data);

	// Set our rendering options to get solid black polygons
	//if (polyAttributes & GENERIC_CEL_SHADE) glDisable(GL_LIGHTING);
	glDisable(GL_LIGHTING);
	glColor3f(0,0,0);
	glDisable(GL_CULL_FACE);
	glDepthFunc(GL_LEQUAL);

	// loop counters & tem variables
	unsigned int x;
	float t, t2;
	vector point1, point2, point3, tempLight;
	tempLight = lightAngle;

	// We start by getting rid of everything OpenGL has done
	// so far (but remember, we saved a backup of the
	// rotation matrix on line 2 of this function)
	glPushMatrix();
 	glLoadIdentity();

	// We begin a triangle list
	glBegin(GL_TRIANGLES);
	// This is a constant for the entirety of the object,
	// so we do it out here
	t = A*tempLight.x + B*tempLight.y + C*tempLight.z;
	for (x=0;x<numberPolys; x++)
	{
		// if t<=0, we are either past the plane or we're casting
		// a shadow paralell to it, so we only go into this
		// when we know we'll hit the plane
		if (t > 0)
		{
			// Copy points from our list
			point1 = genericPolyList[x].p1;
			point2 = genericPolyList[x].p2;
			point3 = genericPolyList[x].p3;

			// Rotate them according to our saved rotation matrix
			point1 = MatrixMult(rotationMatrix, point1);
			point2 = MatrixMult(rotationMatrix, point2);
			point3 = MatrixMult(rotationMatrix, point3);

			// Calculate the "t" that give us an intersection,
			// save in t2
			t2 = -(A*point1.x + B*point1.y + C*point1.z + D) / t;
			// use ray formula to find real world coordinates
			point1.x = point1.x + tempLight.x*t2;
			point1.y = point1.y + tempLight.y*t2;
			point1.z = point1.z + tempLight.z*t2;

			// repeat for point 2 of polygon
			t2 = -(A*point2.x + B*point2.y + C*point2.z + D) / t;
			point2.x = point2.x + tempLight.x*t2;
			point2.y = point2.y + tempLight.y*t2;
			point2.z = point2.z + tempLight.z*t2;

			// and again for point 3 of polygon
			t2 = -(A*point3.x + B*point3.y + C*point3.z + D) / t;
			point3.x = point3.x + tempLight.x*t2;
			point3.y = point3.y + tempLight.y*t2;
			point3.z = point3.z + tempLight.z*t2;

			// and draw a solid black triangle with them
			glVertex3f(point1.x, point1.y, point1.z);
			glVertex3f(point2.x, point2.y, point2.z);
			glVertex3f(point3.x, point3.y, point3.z);
		}
	}
	// Set everything back to normal.
	glEnd();
	glPopMatrix();
	glEnable(GL_CULL_FACE);
	glDepthFunc(GL_LESS);
//	if (polyAttributes & GENERIC_CEL_SHADE) glEnable(GL_LIGHTING);
	glEnable(GL_LIGHTING);
}

void genericModelMesh::LoadModel(char *filename)
{
	ifstream gmmInput(filename);
	int numpolys, attributes, x;
	if (gmmInput.is_open()!=0)
	{
		gmmInput >> numpolys;
		gmmInput >> attributes;
		genericPolyList = new genericPoly[numpolys];
		for (x=0; x<numpolys; x++)
		{
			gmmInput >> genericPolyList[x].p1.x;
			gmmInput >> genericPolyList[x].p1.y;
			gmmInput >> genericPolyList[x].p1.z;
			if (attributes & GENERIC_LOAD_NORMALS)
			{
				gmmInput >> genericPolyList[x].n1.x;
				gmmInput >> genericPolyList[x].n1.y;
				gmmInput >> genericPolyList[x].n1.z;
			}
			if (attributes & GENERIC_LOAD_TEX_COORDS)
			{
				gmmInput >> genericPolyList[x].t1.x;
				gmmInput >> genericPolyList[x].t1.y;
			}

			gmmInput >> genericPolyList[x].p2.x;
			gmmInput >> genericPolyList[x].p2.y;
			gmmInput >> genericPolyList[x].p2.z;
			if (attributes & GENERIC_LOAD_NORMALS)
			{
				gmmInput >> genericPolyList[x].n2.x;
				gmmInput >> genericPolyList[x].n2.y;
				gmmInput >> genericPolyList[x].n2.z;
			}
			if (attributes & GENERIC_LOAD_TEX_COORDS)
			{
				gmmInput >> genericPolyList[x].t2.x;
				gmmInput >> genericPolyList[x].t2.y;
			}
			
			gmmInput >> genericPolyList[x].p3.x;
			gmmInput >> genericPolyList[x].p3.y;
			gmmInput >> genericPolyList[x].p3.z;
			if (attributes & GENERIC_LOAD_NORMALS)
			{
				gmmInput >> genericPolyList[x].n3.x;
				gmmInput >> genericPolyList[x].n3.y;
				gmmInput >> genericPolyList[x].n3.z;
			}
			if (attributes & GENERIC_LOAD_TEX_COORDS)
			{
				gmmInput >> genericPolyList[x].t3.x;
				gmmInput >> genericPolyList[x].t3.y;
			}
		}
	}
	numberPolys = numpolys;
	polyAttributes = attributes;
	if (attributes & GENERIC_LOAD_NORMALS) SetFlag(GENERIC_DRAW_NORM);
	if (attributes & GENERIC_LOAD_TEX_COORDS) SetFlag(GENERIC_DRAW_TEX);
	gmmInput.close();
}

#endif //GENERIC_POLY_H